<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use OpenAI\Laravel\Facades\OpenAI;
use DOMDocument;
use DOMXPath;

class AiChatController extends Controller
{
    private $websiteUrl = 'https://ffgameshop.com';
    
    /**
     * Handle AI chat requests with real-time website analysis
     */
    public function chat(Request $request)
    {
        try {
            $request->validate([
                'message' => 'required|string|max:2000',
            ]);

            $userMessage = $request->input('message');
            $userIp = $request->ip();
            
            // Rate limiting
            $rateLimitKey = 'chat_rate_limit_' . $userIp;
            $requestCount = Cache::get($rateLimitKey, 0);
            
            if ($requestCount >= 25) {
                return response()->json([
                    'error' => 'অনুগ্রহ করে একটু অপেক্ষা করুন। অনেক বেশি অনুরোধ পাঠানো হয়েছে। ⏰'
                ], 429);
            }
            
            Cache::put($rateLimitKey, $requestCount + 1, 60);

            if (!$this->isOpenAIAvailable()) {
                return $this->getFallbackResponse($userMessage, $userIp);
            }

            // Perform real website analysis
            $websiteAnalysis = $this->performRealWebsiteAnalysis($userMessage);
            
            // Get conversation context
            $conversationHistory = $this->getConversationHistory($userIp);
            
            try {
                $aiResponse = $this->callOpenAIWithWebsiteData(
                    $userMessage, 
                    $websiteAnalysis, 
                    $conversationHistory,
                    $userIp
                );
                
                $this->storeConversation($userIp, $userMessage, $aiResponse);
                $this->logInteraction($userMessage, $aiResponse, $userIp, 'success', $websiteAnalysis);

                return response()->json([
                    'reply' => $aiResponse,
                    'timestamp' => now()->toISOString(),
                    'status' => 'success',
                    'website_analyzed' => true,
                    'content_length' => $websiteAnalysis['content_length'] ?? 0,
                    'data_extracted' => count($websiteAnalysis['extracted_info'] ?? [])
                ]);

            } catch (\Exception $apiError) {
                Log::error('OpenAI API Error: ' . $apiError->getMessage());
                return $this->getFallbackResponse($userMessage, $userIp);
            }

        } catch (\Exception $e) {
            Log::error('Chat Controller Error', [
                'message' => $e->getMessage(),
                'user_ip' => $request->ip(),
                'user_message' => $request->input('message', 'N/A')
            ]);

            return $this->getFallbackResponse(
                $request->input('message', ''), 
                $request->ip()
            );
        }
    }

    /**
     * Perform real website analysis
     */
    public function performRealWebsiteAnalysis($userMessage)
    {
        $cacheKey = 'real_website_analysis_' . md5($userMessage . date('i')); // Cache for 1 minute
        
        return Cache::remember($cacheKey, 60, function () use ($userMessage) {
            Log::info('Starting real website analysis', ['message' => $userMessage]);
            
            try {
                // Fetch website content
                $websiteContent = $this->fetchRealWebsiteContent();
                
                if (empty($websiteContent)) {
                    Log::warning('No website content fetched');
                    return $this->getStaticFallbackData();
                }
                
                // Extract information from the content
                $extractedInfo = $this->extractInformationFromContent($websiteContent);
                
                // Analyze user question
                $questionAnalysis = $this->analyzeUserQuestion($userMessage);
                
                // Build context for AI
                $aiContext = $this->buildAIContext($extractedInfo, $questionAnalysis, $userMessage);
                
                $result = [
                    'content_fetched' => true,
                    'content_length' => strlen($websiteContent),
                    'extracted_info' => $extractedInfo,
                    'question_analysis' => $questionAnalysis,
                    'ai_context' => $aiContext,
                    'analysis_time' => now()->toISOString()
                ];
                
                Log::info('Website analysis completed successfully', [
                    'content_length' => $result['content_length'],
                    'extracted_items' => count($extractedInfo),
                    'question_type' => $questionAnalysis['type']
                ]);
                
                return $result;
                
            } catch (\Exception $e) {
                Log::error('Website analysis failed: ' . $e->getMessage());
                return $this->getStaticFallbackData();
            }
        });
    }

    /**
     * Fetch real website content
     */
    private function fetchRealWebsiteContent()
    {
        try {
            Log::info("Fetching real content from: {$this->websiteUrl}");
            
            $response = Http::timeout(20)
                ->withHeaders([
                    'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                    'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                    'Accept-Language' => 'en-US,en;q=0.9,bn;q=0.8',
                    'Accept-Encoding' => 'gzip, deflate, br',
                    'Connection' => 'keep-alive',
                    'Upgrade-Insecure-Requests' => '1',
                    'Cache-Control' => 'no-cache',
                    'Pragma' => 'no-cache'
                ])
                ->get($this->websiteUrl);
            
            if ($response->successful()) {
                $content = $response->body();
                Log::info("Successfully fetched website content", [
                    'length' => strlen($content),
                    'status' => $response->status()
                ]);
                return $content;
            } else {
                Log::warning("Failed to fetch website", [
                    'status' => $response->status(),
                    'response' => substr($response->body(), 0, 200)
                ]);
                return '';
            }
            
        } catch (\Exception $e) {
            Log::error("Error fetching website: " . $e->getMessage());
            return '';
        }
    }

    /**
     * Extract information from website content
     */
    private function extractInformationFromContent($htmlContent)
    {
        $extractedInfo = [];
        
        try {
            // Parse HTML
            $dom = new DOMDocument();
            libxml_use_internal_errors(true);
            @$dom->loadHTML($htmlContent);
            libxml_clear_errors();
            
            $xpath = new DOMXPath($dom);
            
            // Extract title
            $titleNodes = $xpath->query('//title');
            if ($titleNodes->length > 0) {
                $extractedInfo['title'] = trim($titleNodes->item(0)->textContent);
            }
            
            // Extract all text content
            $bodyNodes = $xpath->query('//body');
            if ($bodyNodes->length > 0) {
                $bodyText = $bodyNodes->item(0)->textContent;
                $extractedInfo['full_text'] = $this->cleanText($bodyText);
            }
            
            // Extract pricing information
            $extractedInfo['pricing'] = $this->extractPricingFromText($extractedInfo['full_text'] ?? '');
            
            // Extract contact information
            $extractedInfo['contact'] = $this->extractContactFromText($extractedInfo['full_text'] ?? '');
            
            // Extract product information
            $extractedInfo['products'] = $this->extractProductsFromText($extractedInfo['full_text'] ?? '');
            
            // Extract process information
            $extractedInfo['processes'] = $this->extractProcessFromText($extractedInfo['full_text'] ?? '');
            
            Log::info('Information extraction completed', [
                'title' => $extractedInfo['title'] ?? 'Not found',
                'text_length' => strlen($extractedInfo['full_text'] ?? ''),
                'pricing_items' => count($extractedInfo['pricing']),
                'contact_items' => count($extractedInfo['contact']),
                'product_items' => count($extractedInfo['products'])
            ]);
            
        } catch (\Exception $e) {
            Log::error('Information extraction failed: ' . $e->getMessage());
        }
        
        return $extractedInfo;
    }

    /**
     * Clean text content
     */
    private function cleanText($text)
    {
        // Remove extra whitespace and normalize
        $text = preg_replace('/\s+/', ' ', $text);
        $text = preg_replace('/\n+/', "\n", $text);
        return trim($text);
    }

    /**
     * Extract pricing from text
     */
    private function extractPricingFromText($text)
    {
        $pricing = [];
        
        // Multiple pricing patterns
        $patterns = [
            // Pattern: 115 Diamond 73 টাকা
            '/(\d+)\s*(Diamond|ডায়মন্ড|UC|diamond)\s*[^\d]*(\d+)\s*টাকা/i',
            // Pattern: 73 টাকা 115 Diamond
            '/(\d+)\s*টাকা[^\d]*(\d+)\s*(Diamond|ডায়মন্ড|UC|diamond)/i',
            // Pattern: Diamond 115 = 73 Taka
            '/(Diamond|ডায়মন্ড|UC|diamond)\s*(\d+)[^\d]*(\d+)\s*(টাকা|Taka)/i',
            // Pattern: 115 Diamond (73 টাকা)
            '/(\d+)\s*(Diamond|ডায়মন্ড|UC|diamond)[^\d]*$$(\d+)\s*টাকা$$/i',
            // Pattern: Price: 73 for 115 Diamond
            '/Price[:\s]*(\d+)[^\d]*(\d+)\s*(Diamond|ডায়মন্ড|UC|diamond)/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match_all($pattern, $text, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    if (count($match) >= 4) {
                        // Determine which number is quantity and which is price
                        $quantity = '';
                        $price = '';
                        $currency = '';
                        
                        if (stripos($match[2], 'Diamond') !== false || stripos($match[2], 'ডায়মন্ড') !== false || stripos($match[2], 'UC') !== false) {
                            $quantity = $match[1];
                            $currency = $match[2];
                            $price = $match[3];
                        } elseif (stripos($match[3], 'Diamond') !== false || stripos($match[3], 'ডায়মন্ড') !== false || stripos($match[3], 'UC') !== false) {
                            $price = $match[1];
                            $quantity = $match[2];
                            $currency = $match[3];
                        }
                        
                        if ($quantity && $price && $currency) {
                            $item = $quantity . ' ' . $currency;
                            $pricing[$item] = $price . ' টাকা';
                        }
                    }
                }
            }
        }
        
        // If no pricing found, look for any numbers with Diamond/UC and টাকা
        if (empty($pricing)) {
            if (preg_match_all('/(\d+)[^\d]*(?:Diamond|ডায়মন্ড|UC)[^\d]*(\d+)[^\d]*টাকা/i', $text, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    $pricing[$match[1] . ' Diamond'] = $match[2] . ' টাকা';
                }
            }
        }
        
        return $pricing;
    }

    /**
     * Extract contact information from text
     */
    private function extractContactFromText($text)
    {
        $contact = [];
        
        // Phone numbers (Bangladesh format)
        if (preg_match_all('/(\+88)?0?1[3-9]\d{8}/', $text, $phones)) {
            $contact['phones'] = array_unique($phones[0]);
        }
        
        // Email addresses
        if (preg_match_all('/[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/', $text, $emails)) {
            $contact['emails'] = array_unique($emails[0]);
        }
        
        // WhatsApp
        if (preg_match_all('/(?:whatsapp|হোয়াটসঅ্যাপ)[^\d]*(\+?88)?0?1[3-9]\d{8}/i', $text, $whatsapp)) {
            $contact['whatsapp'] = array_unique($whatsapp[0]);
        }
        
        return $contact;
    }

    /**
     * Extract products from text
     */
    private function extractProductsFromText($text)
    {
        $products = [];
        
        $gameKeywords = [
            'Free Fire' => ['Free Fire', 'FreeFire', 'ফ্রি ফায়ার'],
            'PUBG Mobile' => ['PUBG Mobile', 'PUBG', 'পাবজি'],
            'Mobile Legends' => ['Mobile Legends', 'ML', 'মোবাইল লিজেন্ড'],
            'Call of Duty' => ['Call of Duty', 'COD', 'কল অফ ডিউটি'],
            'Clash of Clans' => ['Clash of Clans', 'COC', 'ক্ল্যাশ অফ ক্ল্যান']
        ];
        
        foreach ($gameKeywords as $game => $keywords) {
            foreach ($keywords as $keyword) {
                if (stripos($text, $keyword) !== false) {
                    $products[] = $game;
                    break;
                }
            }
        }
        
        // Currency types
        if (stripos($text, 'Diamond') !== false || stripos($text, 'ডায়মন্ড') !== false) {
            $products[] = 'Diamond TopUp';
        }
        if (stripos($text, 'UC') !== false) {
            $products[] = 'UC TopUp';
        }
        
        return array_unique($products);
    }

    /**
     * Extract process information from text
     */
    private function extractProcessFromText($text)
    {
        $processes = [];
        
        // Look for step-by-step instructions
        $stepPatterns = [
            '/(?:ধাপ|Step)\s*\d+[:\-\s]*([^।\n\.]+)/i',
            '/\d+\.\s*([^।\n\.]+)/i',
            '/(?:প্রথমে|First|শুরুতে)[:\-\s]*([^।\n\.]+)/i',
            '/(?:তারপর|Then|এরপর)[:\-\s]*([^।\n\.]+)/i'
        ];
        
        foreach ($stepPatterns as $pattern) {
            if (preg_match_all($pattern, $text, $matches)) {
                foreach ($matches[1] as $step) {
                    $step = trim($step);
                    if (strlen($step) > 10 && strlen($step) < 200) {
                        $processes[] = $step;
                    }
                }
            }
        }
        
        return array_unique($processes);
    }

    /**
     * Analyze user question
     */
    private function analyzeUserQuestion($userMessage)
    {
        $message = strtolower($userMessage);
        
        $analysis = [
            'type' => 'general',
            'intent' => 'information',
            'keywords' => [],
            'specific_items' => []
        ];
        
        // Determine question type
        if (preg_match('/(দাম|price|cost|টাকা|কত|খরচ)/i', $message)) {
            $analysis['type'] = 'pricing';
            $analysis['intent'] = 'price_inquiry';
        } elseif (preg_match('/(কিভাবে|how|order|অর্ডার|process|পদ্ধতি)/i', $message)) {
            $analysis['type'] = 'process';
            $analysis['intent'] = 'process_inquiry';
        } elseif (preg_match('/(contact|যোগাযোগ|help|সাহায্য|support|নাম্বার|number)/i', $message)) {
            $analysis['type'] = 'contact';
            $analysis['intent'] = 'contact_inquiry';
        } elseif (preg_match('/(game|গেম|diamond|ডায়মন্ড|uc|product)/i', $message)) {
            $analysis['type'] = 'product';
            $analysis['intent'] = 'product_inquiry';
        }
        
        // Extract specific items mentioned
        if (preg_match_all('/\d+/', $userMessage, $numbers)) {
            $analysis['specific_items']['numbers'] = $numbers[0];
        }
        
        if (preg_match('/(diamond|ডায়মন্ড|uc|free fire|pubg)/i', $userMessage, $games)) {
            $analysis['specific_items']['games'] = $games[0];
        }
        
        return $analysis;
    }

    /**
     * Build AI context
     */
    private function buildAIContext($extractedInfo, $questionAnalysis, $userMessage)
    {
        $context = "=== FF GAMESHOP ওয়েবসাইট বিশ্লেষণ (রিয়েল-টাইম) ===\n\n";
        
        // Add title
        if (!empty($extractedInfo['title'])) {
            $context .= "🌐 ওয়েবসাইট: {$extractedInfo['title']}\n\n";
        }
        
        // Add relevant information based on question type
        switch ($questionAnalysis['type']) {
            case 'pricing':
                if (!empty($extractedInfo['pricing'])) {
                    $context .= "💰 বর্তমান দামের তালিকা:\n";
                    foreach ($extractedInfo['pricing'] as $item => $price) {
                        $context .= "• $item = $price\n";
                    }
                    $context .= "\n";
                }
                break;
                
            case 'contact':
                if (!empty($extractedInfo['contact'])) {
                    $context .= "📞 যোগাযোগের তথ্য:\n";
                    if (!empty($extractedInfo['contact']['phones'])) {
                        foreach ($extractedInfo['contact']['phones'] as $phone) {
                            $context .= "• ফোন: $phone\n";
                        }
                    }
                    if (!empty($extractedInfo['contact']['emails'])) {
                        foreach ($extractedInfo['contact']['emails'] as $email) {
                            $context .= "• ইমেইল: $email\n";
                        }
                    }
                    $context .= "\n";
                }
                break;
                
            case 'product':
                if (!empty($extractedInfo['products'])) {
                    $context .= "🎮 উপলব্ধ পণ্য/সেবা:\n";
                    foreach ($extractedInfo['products'] as $product) {
                        $context .= "• $product\n";
                    }
                    $context .= "\n";
                }
                break;
                
            case 'process':
                if (!empty($extractedInfo['processes'])) {
                    $context .= "📋 প্রক্রিয়া/নির্দেশনা:\n";
                    foreach (array_slice($extractedInfo['processes'], 0, 5) as $process) {
                        $context .= "• $process\n";
                    }
                    $context .= "\n";
                }
                break;
        }
        
        // Add all available information for comprehensive context
        $context .= "📊 সম্পূর্ণ তথ্য:\n";
        
        if (!empty($extractedInfo['pricing'])) {
            $context .= "দাম: " . count($extractedInfo['pricing']) . " টি আইটেম পাওয়া গেছে\n";
        }
        
        if (!empty($extractedInfo['products'])) {
            $context .= "পণ্য: " . implode(', ', $extractedInfo['products']) . "\n";
        }
        
        if (!empty($extractedInfo['contact']['phones'])) {
            $context .= "ফোন: " . implode(', ', $extractedInfo['contact']['phones']) . "\n";
        }
        
        return $context;
    }

    /**
     * Get static fallback data
     */
    private function getStaticFallbackData()
    {
        return [
            'content_fetched' => false,
            'content_length' => 0,
            'extracted_info' => [
                'title' => 'FF GAMESHOP',
                'pricing' => [
                    '115 Diamond' => '73 টাকা',
                    '240 Diamond' => '152 টাকা'
                ],
                'contact' => [
                    'phones' => ['+8801609177188']
                ],
                'products' => ['Free Fire Diamond', 'PUBG UC']
            ],
            'question_analysis' => ['type' => 'general'],
            'ai_context' => 'Static fallback data used',
            'analysis_time' => now()->toISOString(),
            'fallback_mode' => true
        ];
    }

    /**
     * Call OpenAI with website data
     */
    private function callOpenAIWithWebsiteData($userMessage, $websiteAnalysis, $conversationHistory, $userIp)
    {
        $messages = [
            [
                'role' => 'system',
                'content' => $this->buildSystemPromptWithRealData($websiteAnalysis)
            ]
        ];

        // Add conversation history
        foreach (array_slice($conversationHistory, -2) as $conversation) {
            $messages[] = [
                'role' => 'user',
                'content' => $conversation['user_message']
            ];
            $messages[] = [
                'role' => 'assistant', 
                'content' => $conversation['ai_response']
            ];
        }

        // Add current message
        $messages[] = [
            'role' => 'user',
            'content' => $userMessage
        ];

        $response = OpenAI::chat()->create([
            'model' => 'gpt-4o-mini',
            'messages' => $messages,
            'max_tokens' => 800,
            'temperature' => 0.95, // High for varied responses
            'frequency_penalty' => 1.0, // Maximum to avoid repetition
            'presence_penalty' => 0.8,
            'user' => $userIp,
        ]);

        return $this->processAiResponse($response->choices[0]->message->content);
    }

    /**
     * Build system prompt with real data
     */
    private function buildSystemPromptWithRealData($websiteAnalysis)
    {
        $aiContext = $websiteAnalysis['ai_context'];
        $questionType = $websiteAnalysis['question_analysis']['type'];
        $contentFetched = $websiteAnalysis['content_fetched'] ? 'হ্যাঁ' : 'না';
        
        return "
আপনি FF GAMESHOP এর একজন বিশেষজ্ঞ AI সহায়ক। আপনি এইমাত্র ওয়েবসাইট বিশ্লেষণ করেছেন।

🔍 **বিশ্লেষণের ফলাফল:**
• ওয়েবসাইট কন্টেন্ট সংগ্রহ: $contentFetched
• প্রশ্নের ধরন: $questionType
• কন্টেন্ট দৈর্ঘ্য: {$websiteAnalysis['content_length']} অক্ষর

$aiContext

🎯 **আপনার দায়িত্ব:**
✅ উপরের রিয়েল-টাইম ওয়েবসাইট ডেটা ব্যবহার করে উত্তর দিন
✅ প্রতিটি উত্তর অবশ্যই অনন্য এবং প্রশ্ন-নির্দিষ্ট হতে হবে
✅ সঠিক দাম, ফোন নাম্বার এবং তথ্য দিন
✅ গ্রাহকের প্রশ্নের সরাসরি উত্তর দিন
✅ বন্ধুত্বপূর্ণ এবং সহায়ক ভাষা ব্যবহার করুন
✅ প্রয়োজনে ইমোজি ব্যবহার করুন

🚫 **কঠোরভাবে এড়িয়ে চলুন:**
❌ একই ধরনের উত্তর বারবার দেওয়া
❌ অনুমানভিত্তিক তথ্য দেওয়া  
❌ ওয়েবসাইটে নেই এমন তথ্য দেওয়া
❌ সাধারণ বা অস্পষ্ট উত্তর

এখন গ্রাহকের প্রশ্নের একটি নির্দিষ্ট, সঠিক এবং সহায়ক উত্তর দিন যা সম্পূর্ণভাবে ওয়েবসাইট বিশ্লেষণের উপর ভিত্তি করে।
        ";
    }

    /**
     * Check if OpenAI is available
     */
    private function isOpenAIAvailable()
    {
        if (!class_exists('\OpenAI\Laravel\Facades\OpenAI')) {
            return false;
        }

        $apiKey = config('openai.api_key') ?? env('OPENAI_API_KEY');
        return !empty($apiKey);
    }

    /**
     * Process AI response
     */
    private function processAiResponse($response)
    {
        $response = trim($response);
        
        // Add contextual emojis
        $emojiPatterns = [
            '/ধন্যবাদ(?![🙏])/' => 'ধন্যবাদ 🙏',
            '/স্বাগতম(?![🤗])/' => 'স্বাগতম 🤗',
            '/সাহায্য(?![🤝])/' => 'সাহায্য 🤝',
            '/অর্ডার(?![📦])/' => 'অর্ডার 📦',
            '/পেমেন্ট(?![💳])/' => 'পেমেন্ট 💳',
            '/ডায়মন্ড(?![💎])/' => 'ডায়মন্ড 💎',
            '/টাকা(?![💰])/' => 'টাকা 💰',
            '/দ্রুত(?![⚡])/' => 'দ্রুত ⚡',
            '/নিরাপদ(?![🛡️])/' => 'নিরাপদ 🛡️',
        ];

        foreach ($emojiPatterns as $pattern => $replacement) {
            $response = preg_replace($pattern, $replacement, $response, 1);
        }

        return $response;
    }

    /**
     * Get conversation history
     */
    private function getConversationHistory($userIp)
    {
        $cacheKey = 'conversation_history_' . md5($userIp);
        return Cache::get($cacheKey, []);
    }

    /**
     * Store conversation
     */
    private function storeConversation($userIp, $userMessage, $aiResponse)
    {
        $cacheKey = 'conversation_history_' . md5($userIp);
        $history = Cache::get($cacheKey, []);
        
        $history[] = [
            'user_message' => $userMessage,
            'ai_response' => $aiResponse,
            'timestamp' => now()->toISOString()
        ];
        
        // Keep only last 3 conversations
        if (count($history) > 3) {
            $history = array_slice($history, -3);
        }
        
        Cache::put($cacheKey, $history, 1800);
    }

    /**
     * Get fallback response
     */
    private function getFallbackResponse($userMessage, $userIp)
    {
        $fallbackResponse = $this->getIntelligentFallback($userMessage);
        
        $this->logInteraction($userMessage, $fallbackResponse, $userIp, 'fallback');
        
        return response()->json([
            'reply' => $fallbackResponse,
            'timestamp' => now()->toISOString(),
            'status' => 'fallback'
        ]);
    }

    /**
     * Intelligent fallback responses
     */
    private function getIntelligentFallback($userMessage)
    {
        $message = strtolower($userMessage);
        
        if (strpos($message, 'diamond') !== false || strpos($message, 'ডায়মন্ড') !== false || strpos($message, 'দাম') !== false) {
            return "💎 **Free Fire Diamond এর বর্তমান দাম:**

• 115 Diamond = 73 টাকা ⚡
• 240 Diamond = 152 টাকা
• 610 Diamond = 371 টাকা  
• 1080 Diamond = 850 টাকা
• 2530 Diamond = 1520 টাকা

🎯 **অর্ডার করতে:** ffgameshop.com
⚡ **ডেলিভারি:** 1-5 মিনিট
📞 **সাহায্য:** +8801609177188";
        }
        
        if (strpos($message, 'যোগাযোগ') !== false || strpos($message, 'contact') !== false || strpos($message, 'নাম্বার') !== false) {
            return "📞 **FF GAMESHOP যোগাযোগের তথ্য:**

• **প্রধান ফোন:** +8801609177188
• **বিকল্প ফোন:** +8801400100956  
• **হোয়াটসঅ্যাপ:** +8801609177188
• **ওয়েবসাইট:** ffgameshop.com

⏰ **সেবার সময়:** ২৪/৭ উপলব্ধ
🤝 **সাহায্য:** যেকোনো সময় যোগাযোগ করুন!";
        }
        
        return "আসসালামু আলাইকুম! 👋 

আমি **FF GAMESHOP** এর AI সহায়ক। আমি রিয়েল-টাইমে ওয়েবসাইট বিশ্লেষণ করে আপনাকে সাহায্য করি।

🔍 **আমি যা করতে পারি:**
• সর্বশেষ দাম ও অফার জানাতে
• অর্ডার প্রক্রিয়া গাইড করতে  
• যোগাযোগের তথ্য দিতে
• যেকোনো প্রশ্নের উত্তর দিতে

আপনার প্রশ্ন করুন! 😊";
    }

    /**
     * Log interactions
     */
    private function logInteraction($userMessage, $aiResponse, $userIp, $status, $websiteAnalysis = null)
    {
        $logData = [
            'user_message' => substr($userMessage, 0, 100),
            'ai_response_length' => $aiResponse ? strlen($aiResponse) : 0,
            'user_ip' => $userIp,
            'status' => $status,
            'timestamp' => now(),
        ];

        if ($websiteAnalysis) {
            $logData['website_analysis'] = [
                'content_fetched' => $websiteAnalysis['content_fetched'] ?? false,
                'content_length' => $websiteAnalysis['content_length'] ?? 0,
                'question_type' => $websiteAnalysis['question_analysis']['type'] ?? 'unknown'
            ];
        }

        Log::info('AI Chat Interaction', $logData);
    }
}
